' filepath: c:\Users\dpascoal\Documents\Projects\DHLAllinone\src\scripts\sap-script-labels.vbs
'
' SAP Label Printing Automation Script
' Author: Dario Pascoal
'
' Description: Automated barcode label printing system that integrates with SAP GUI
' to generate and print equipment labels through the ZBCP transaction. This script
' handles the complete workflow from barcode validation to label generation,
' supporting configurable quantities and printer selection for warehouse operations.
'
' The script automates the complex process of accessing SAP's label printing system,
' entering barcode information, configuring print settings, and executing the print
' job. It includes comprehensive error handling, logging capabilities, and security
' configuration management to ensure reliable unattended operation.
'
' Functionality:
' - Connects to active SAP GUI session for automated interaction
' - Validates and processes barcode identifiers for equipment labeling
' - Configures label templates and printer settings automatically
' - Executes print jobs with specified quantities and formats
' - Provides detailed logging for audit trails and troubleshooting
' - Handles SAP security dialogs and authentication automatically
'
' Prerequisites:
' - Active SAP GUI session with user authentication
' - SAP GUI Scripting enabled and properly configured
' - Access to ZBCP transaction for label printing operations
' - Windows Script Host (WSH) environment available
' - Appropriate SAP user permissions for label printing
'
' Parameters:
' - Arg 0: Barcode ID (string) - Equipment identifier for label generation
' - Arg 1: Printer Name (string) - Target printer (currently hardcoded to ZLOCL_TSC_SF600)
' - Arg 2: Quantity (integer) - Number of labels to print
' - Arg 3: Log File Path (optional string) - Path for operation logging
'
' Returns:
' - Console output with operation status and completion messages
' - Log file entries with detailed operation tracking (if path provided)
' - SAP GUI navigation back to starting transaction state
'
' Error Handling:
' - Comprehensive error catching for SAP connection issues
' - Graceful handling of security dialog automation
' - Log file creation and access error management
' - Registry modification error handling for SAP scripting
'
' Maintenance Notes:
' - Printer configuration is currently hardcoded to ZLOCL_TSC_SF600
' - Label template ID is set to "15" for standard equipment labels
' - Security registry modifications require appropriate user permissions
' - SAP GUI element IDs may change with SAP system updates

' =============================================================================
' ENVIRONMENT INITIALIZATION AND ERROR HANDLING
' =============================================================================
' Configure script execution environment with comprehensive error handling
' and security settings for reliable SAP GUI automation

' Enable comprehensive error handling for graceful failure management
' This allows the script to continue operation even when non-critical errors occur
On Error Resume Next

' =============================================================================
' WINDOWS SHELL AND REGISTRY CONFIGURATION
' =============================================================================
' Initialize Windows Shell object for system-level operations and SAP GUI
' scripting security configuration through registry modifications

' Create Windows Shell object for system interaction and registry access
' This object provides access to Windows registry and system automation features
Set WSHShell = CreateObject("WScript.Shell")

' =============================================================================
' SAP GUI SECURITY CONFIGURATION
' =============================================================================
' Disable SAP GUI scripting security warnings to enable automated interaction
' These registry modifications are essential for unattended script execution
' and prevent user prompts that would interrupt the automation workflow

' Disable attachment warnings when scripts connect to SAP GUI sessions
' This prevents popup dialogs that would halt the label printing automation
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"

' Disable connection warnings for script-to-SAP communication
' This ensures seamless connection establishment without user intervention
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"

' Disable allow list requirement warnings for automated scripting
' This prevents security prompts that would require manual user interaction
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"

' Disable allow list usage to prevent script blocking
' This allows our automation script to execute without whitelist restrictions
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' =============================================================================
' LOGGING SYSTEM INITIALIZATION
' =============================================================================
' Setup comprehensive logging system for operation tracking and debugging
' The logging system supports detailed file logging with timestamps for
' audit trails, troubleshooting, and operational monitoring

' Declare log file objects and path variables for file-based logging
' These objects handle all file-based logging operations throughout the script
Dim logFile, logFilePath

' =============================================================================
' COMMAND LINE ARGUMENT PROCESSING
' =============================================================================
' Extract and validate command line arguments for barcode processing, printer
' configuration, and print quantity settings for the label printing operation

' Extract barcode identifier from first command line argument (required)
' This barcode ID identifies the equipment for which labels will be generated
barcodeId = WScript.Arguments.Item(0)

' Extract printer name from second command line argument (for future use)
' Currently hardcoded to ZLOCL_TSC_SF600 but maintained for configuration flexibility
printerName = WScript.Arguments.Item(1)

' Extract print quantity from third command line argument (required)
' This determines how many copies of the label will be printed
quantity = WScript.Arguments.Item(2)

' =============================================================================
' OPTIONAL LOG FILE INITIALIZATION
' =============================================================================
' Initialize file-based logging system if log file path is provided as
' fourth command line argument for detailed operation tracking

' Initialize log file if path is provided as optional fourth argument
' This enables detailed file-based logging for audit trails and debugging
If WScript.Arguments.Count > 3 Then
    logFilePath = WScript.Arguments.Item(3)
    
    ' Attempt to open the log file in append mode for continuous logging
    ' Append mode (8) allows multiple script executions to write to same log
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle log file creation or access errors gracefully
    ' If log file cannot be opened, continue operation without file logging
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Write script initialization header to log for session identification
        ' This creates a clear audit trail for each script execution
        LogToFile "========================================="
        LogToFile "Labels Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Barcode ID: " & barcodeId
        LogToFile "Printer: " & printerName
        LogToFile "Quantity: " & quantity
        LogToFile "========================================="
    End If
End If

' =============================================================================
' LOGGING UTILITY FUNCTIONS
' =============================================================================
' Comprehensive logging functions that support both console output and
' file-based logging for complete operation visibility and audit trails

'
' Function to write timestamped messages to log file for audit trail maintenance
' This function provides detailed file-based logging with timestamps for
' troubleshooting, compliance, and operational monitoring purposes
'
' Parameters:
' - message: String message to be logged with timestamp and script identification
'
Sub LogToFile(message)
    On Error Resume Next
    ' Check if log file object is available and properly initialized
    If Not logFile Is Nothing Then
        ' Write formatted log entry with timestamp and script identification
        ' Format: [Timestamp] - [Script Name] - [Message]
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

'
' Function to output messages to both console and log file for comprehensive logging
' This dual-output approach ensures immediate visibility while maintaining
' permanent audit trails for compliance and troubleshooting purposes
'
' Parameters:
' - message: String message to be displayed and logged
'
Sub LogMessage(message)
    ' Display message immediately in console for real-time monitoring
    WScript.Echo message
    
    ' Also write message to log file for permanent audit trail
    LogToFile message
End Sub

' =============================================================================
' SAP GUI CONNECTION ESTABLISHMENT
' =============================================================================
' Establish connection to SAP GUI application and active session for
' automated interaction with SAP's label printing functionality

' Initialize SAP GUI application object if not already connected
' The application object is the top-level SAP GUI automation interface
If Not IsObject(application) Then
   ' Connect to the SAP GUI Automation Server for script access
   ' This server manages all SAP GUI automation and scripting operations
   Set SapGuiAuto = GetObject("SAPGUI")
   
   ' Get the scripting engine from the SAP GUI server
   ' The scripting engine provides programmatic access to SAP GUI elements
   Set application = SapGuiAuto.GetScriptingEngine
End If

' Initialize SAP connection object if not already established
' The connection object represents a specific SAP system connection
If Not IsObject(connection) Then
   ' Get the first (primary) connection from available connections
   ' This assumes the user has an active SAP connection established
   Set connection = application.Children(0)
End If

' Initialize SAP session object for transaction execution
' The session object represents the active user session for automation
If Not IsObject(session) Then
   ' Get the first (active) session from the established connection
   ' This session will be used for all SAP transaction automation
   Set session = connection.Children(0)
End If

' Enable event handling for SAP GUI automation if Windows Script Host is available
' This allows the script to respond to SAP GUI events and maintain connection state
If IsObject(WScript) Then
   ' Connect session object to handle SAP GUI session events
   WScript.ConnectObject session, "on"
   
   ' Connect application object to handle SAP GUI application events
   WScript.ConnectObject application, "on"
End If

' =============================================================================
' SECURITY DIALOG AUTOMATION
' =============================================================================
' Automated handling of SAP GUI security dialogs that may appear during
' script execution, ensuring unattended operation without user intervention

'
' Handle security dialog automation for unattended script execution
' This function automatically dismisses security warnings that could
' interrupt the label printing automation workflow and require manual intervention
'
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Wait up to 10 seconds for security dialog to appear
    ' This prevents infinite waiting while allowing time for dialog display
    Do While timeout < 10
        ' Check if SAP GUI Security dialog is active
        ' AppActivate returns True if the specified window title is found
        If shell.AppActivate("SAP GUI Security") Then
            ' Brief pause to ensure dialog is fully loaded and ready for input
            ' This prevents sending keys before dialog elements are interactive
            WScript.Sleep 500
            
            ' Send space key to check "Remember my decision" checkbox
            ' This prevents future security dialogs for this script execution
            shell.SendKeys " "
            WScript.Sleep 200  ' Brief pause between key operations
            
            ' Send Enter key to click "Allow" button and dismiss dialog
            ' This grants permission for script to interact with SAP GUI
            shell.SendKeys "{ENTER}"
            dialogFound = True
            Exit Do
        End If
        
        ' Wait 500ms before checking again for dialog appearance
        ' This provides reasonable response time without excessive CPU usage
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' =============================================================================
' MAIN LABEL PRINTING WORKFLOW EXECUTION
' =============================================================================
' Execute the complete label printing workflow through SAP's ZBCP transaction
' including barcode entry, printer configuration, and print job submission

' Execute security dialog handling before starting SAP operations
' This ensures any security prompts are automatically handled without user intervention
HandleSecurityDialog()

' =============================================================================
' SAP WINDOW AND TRANSACTION PREPARATION
' =============================================================================
' Prepare SAP GUI window and navigate to label printing transaction for
' optimal automation visibility and reliable element interaction

' Maximize SAP window for optimal visibility and element access
' This ensures all GUI elements are visible and accessible for automation
session.findById("wnd[0]").maximize

' Navigate to ZBCP transaction for barcode label printing operations
' ZBCP is the custom SAP transaction for equipment label generation and printing
session.findById("wnd[0]/tbar[0]/okcd").text = "zbcp"
session.findById("wnd[0]").sendVKey 0

' =============================================================================
' LABEL CONFIGURATION AND PRINT EXECUTION
' =============================================================================
' Configure label parameters including barcode data, quantity, template
' selection, and printer assignment for automated label generation

' Enter barcode identifier in the designated input field
' This barcode ID identifies the specific equipment for label generation
session.findById("wnd[0]/usr/ctxtGV_BARCODE").text = barcodeId

' Set print quantity for the number of labels to be generated
' This determines how many copies of the equipment label will be printed
session.findById("wnd[0]/usr/txtGV_QUANTITY").text = quantity

' Select label template definition using predefined key "15"
' This template key corresponds to the standard equipment label format
session.findById("wnd[0]/usr/cmbZBC_LBL_DEF-BCLABELID").key = "15"

' Configure printer selection using hardcoded ZLOCL_TSC_SF600 printer
' This printer is the designated label printer for equipment labels
session.findById("wnd[0]/usr/cmbZBC_LBL_PRT-NAME").key = "ZLOCL_TSC_SF600"

' Set cursor position in barcode field for proper field validation
' This ensures the barcode field is properly focused and validated
session.findById("wnd[0]/usr/ctxtGV_BARCODE").caretPosition = 10

' Execute the label printing operation by sending Enter key (VKey 0)
' This submits the label printing request to SAP's printing subsystem
session.findById("wnd[0]").sendVKey 0

' =============================================================================
' OPERATION COMPLETION AND SUCCESS LOGGING
' =============================================================================
' Log successful completion of label printing operation and return SAP
' to previous state for subsequent operations or user interaction

' Log successful completion of label printing operation
' This provides confirmation that labels were successfully submitted for printing
LogMessage "Labels printed successfully"

' Navigate back to previous SAP screen using back button
' This returns SAP to the state before the ZBCP transaction was executed
session.findById("wnd[0]/tbar[0]/btn[3]").press

' =============================================================================
' CLEANUP AND FINALIZATION PROCEDURES
' =============================================================================
' Comprehensive cleanup procedures to ensure proper resource management
' and logging finalization before script termination

'
' Cleanup function to properly close log file and release system resources
' This function ensures proper log file closure and resource cleanup to
' prevent memory leaks and ensure all log data is properly written to disk
'
Sub CloseLogFile()
    On Error Resume Next
    ' Check if log file is still open and available for closure
    If Not logFile Is Nothing Then
        ' Write session end marker to log for clear audit trail separation
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        
        ' Properly close the log file to ensure all buffered data is written
        logFile.Close
        
        ' Release the log file object to free system resources
        Set logFile = Nothing
    End If
End Sub

' Execute final cleanup procedures before script termination
' This ensures proper resource cleanup and log file closure in all cases
CloseLogFile()